import 'package:flutter/material.dart';
import 'package:get/get.dart';
import 'package:viserpay_agent/core/helper/date_converter.dart';
import 'package:viserpay_agent/core/helper/string_format_helper.dart';
import 'package:viserpay_agent/core/utils/dimensions.dart';
import 'package:viserpay_agent/core/utils/my_color.dart';
import 'package:viserpay_agent/core/utils/my_strings.dart';
import 'package:viserpay_agent/core/utils/style.dart';
import 'package:viserpay_agent/data/controller/commission/commission_log_controller.dart';
import 'package:viserpay_agent/data/repo/commission/commission_log_repo.dart';
import 'package:viserpay_agent/data/services/api_service.dart';
import 'package:viserpay_agent/view/components/app-bar/custom_appbar.dart';
import 'package:viserpay_agent/view/components/column_widget/card_column.dart';
import 'package:viserpay_agent/view/components/custom_loader/custom_loader.dart';
import 'package:viserpay_agent/view/components/divider/custom_divider.dart';
import 'package:viserpay_agent/view/components/no_data.dart';

class CommissionLogScreen extends StatefulWidget {
  const CommissionLogScreen({super.key});

  @override
  State<CommissionLogScreen> createState() => _CommissionLogScreenState();
}

class _CommissionLogScreenState extends State<CommissionLogScreen> {
  final ScrollController scrollController = ScrollController();

  void scrollListener() {
    if (scrollController.position.pixels == scrollController.position.maxScrollExtent) {
      if (Get.find<CommissionLogController>().hasNext()) {
        Get.find<CommissionLogController>().loadPaginationData();
      }
    }
  }

  @override
  void initState() {
    Get.put(ApiClient(sharedPreferences: Get.find()));
    Get.put(CommissionLogRepo(apiClient: Get.find()));
    final controller = Get.put(CommissionLogController(commissionLogRepo: Get.find()));
    super.initState();

    WidgetsBinding.instance.addPostFrameCallback((timeStamp) {
      controller.initialSelectedValue();
      scrollController.addListener(scrollListener);
    });
  }

  @override
  void dispose() {
    scrollController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    return GetBuilder<CommissionLogController>(
      builder: (controller) => SafeArea(
        child: Scaffold(
          backgroundColor: MyColor.screenBgColor,
          appBar: CustomAppBar(
            title: MyStrings.commissionLog.tr,
          ),
          body: controller.isLoading ?
          const CustomLoader() :
          SingleChildScrollView(
            controller: scrollController,
            padding: Dimensions.screenPaddingHV,
            physics: const BouncingScrollPhysics(),
            child: controller.commissionList.isEmpty
              ? const Center(
                  child: NoDataWidget(),
                )
              : ListView.separated(
                    shrinkWrap: true,
                    scrollDirection: Axis.vertical,
                    physics: const NeverScrollableScrollPhysics(),
                    padding: EdgeInsets.zero,
                    itemCount: controller.commissionList.length + 1,
                    separatorBuilder: (context, index) => const SizedBox(height: Dimensions.space10),
                    itemBuilder: (context, index) {
                      if (controller.commissionList.length == index) {
                        return controller.hasNext() ? const CustomLoader(isPagination: true) : const SizedBox();
                      }
                      return Container(
                        width: MediaQuery.of(context).size.width,
                        padding: const EdgeInsets.all(Dimensions.space15),
                        decoration: BoxDecoration(color: MyColor.getCardBgColor(), borderRadius: BorderRadius.circular(Dimensions.defaultRadius)),
                        child: Column(
                          crossAxisAlignment: CrossAxisAlignment.start,
                          children: [
                            Row(
                              mainAxisAlignment: MainAxisAlignment.spaceBetween,
                              children: [
                                CardColumn(header: MyStrings.trx.tr, body: controller.commissionList[index].trx ?? ""),
                                CardColumn(
                                  alignmentEnd: true,
                                  header: MyStrings.date.tr,
                                  body: DateConverter.isoStringToLocalDateOnly(controller.commissionList[index].createdAt ?? ""),
                                )
                              ],
                            ),
                            const SizedBox(height: Dimensions.space10),
                            Row(
                              mainAxisAlignment: MainAxisAlignment.spaceBetween,
                              children: [
                                CardColumn(
                                  alignmentEnd: false,
                                  header: MyStrings.operationType.tr,
                                  body: controller.commissionList[index].details ?? "",
                                ),
                                CardColumn(
                                  alignmentEnd: true,
                                  header: MyStrings.postBalance.tr,
                                  body: "${Converter.formatNumber(controller.commissionList[index].postBalance ?? "")} "
                                      "${controller.currency}",
                                ),
                              ],
                            ),
                            const CustomDivider(space: Dimensions.space15),
                            Container(
                              width: MediaQuery.of(context).size.width,
                              padding: const EdgeInsets.all(Dimensions.space10),
                              decoration: BoxDecoration(borderRadius: BorderRadius.circular(Dimensions.defaultRadius), color: MyColor.getScreenBgColor(), border: Border.all(color: MyColor.colorGrey.withOpacity(0.2), width: 0.5)),
                              child: Row(
                                mainAxisAlignment: MainAxisAlignment.spaceBetween,
                                children: [
                                  Text(
                                    MyStrings.amount.tr,
                                    style: semiBoldDefault.copyWith(color: MyColor.colorBlack),
                                  ),
                                  Text(
                                    "${Converter.formatNumber(controller.commissionList[index].amount ?? "")} "
                                    "${controller.currency}",
                                    style: regularDefault.copyWith(color: MyColor.colorGreen, fontWeight: FontWeight.w600),
                                  )
                                ],
                              ),
                            ),
                          ],
                        ),
                      );
                    }),
          ),
        ),
      ),
    );
  }
}
